<?php
/**
 * Mail Mint Pro
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2025-01-07 15:33:17
 * @modify date 2025-01-07 15:33:17
 * @package MintMailPro\App\Admin\API\Actions\Connector
 */
namespace MintMailPro\App\Admin\API\Actions\Connector;

/**
 * Connector handler for WPForms.
 *
 * This class handles API requests related to Contact Form 7 forms, including fetching available forms
 * and their fields. It extends the `BaseConnectorHandler` to implement specific functionality for Contact Form 7.
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector
 * @since 1.17.4
 */
class WpFormsConnectorHandler extends BaseConnectorHandler {

    /**
     * Retrieves a list of WPForms forms.
     *
     * Fetches all Contact Wpforms forms based on the search parameter. Returns a formatted array
     * containing the form ID and title for each matching form.
     *
     * @param array $params An array of parameters for the request.
     *
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
    public function get($params) {
        $search           = isset($params['search']) ? $params['search'] : '';
        $formatted_forms = array();

        // Get all WPForms forms.
        $posts = get_posts(
            array(
                'post_type'   => 'wpforms',
                'numberposts' => -1,
                'orderby'     => 'created_at',
                'order'       => 'DESC',
                'post_status' => 'publish',
                's'           => $search,
            )
        );

        if (is_array($posts) && !empty($posts)) {
            foreach ($posts as $post) {
                if (stripos($post->post_title, $search) !== false) {
                    $formatted_forms[] = array(
                        'value' => $post->ID,
                        'label' => $post->post_title,
                    );
                }
            }
        }

        $response['success'] = true;
        $response['forms']   = $formatted_forms;
        return rest_ensure_response($response);
    }

    /**
     * Retrieves fields from a specific WPForms form.
     *
     * Scans the specified WPForms form for available fields and returns a formatted array
     * of field names and labels, excluding submit buttons and file upload fields.
     *
     * @param array $params An array of parameters for the request, including the form ID.
     *
     * @return array The response containing the list of fields.
     * @since 1.17.4
     */
    public function fields($params){
        $form_id = isset( $params['id'] ) ? sanitize_text_field( $params['id'] ) : '';
        $form = wpforms()->form->get($form_id);
    
        if (!$form) {
            return [];
        }
    
        if (function_exists('wpforms_decode')) {
            $form_content = wpforms_decode($form->post_content);
        } else {
            return [];
        }
        $fields_data  = [];
        if (empty($form_content['fields'])) {
            return $fields_data;
        }
    
        foreach ($form_content['fields'] as $field) {
            if (!empty($field['isHidden']) || $field['type'] === 'submit') {
                continue;
            }
    
            if (!empty($field['columns'])) {
                $column_fields = self::process_wp_form_column_fields($field['columns']);
                $fields_data   = array_merge($fields_data, $column_fields);
                continue;
            }
    
            $fields_data[] = [
                'type'  => $field['type'],
                'label' => !empty($field['label']) ? $field['label'] : ucfirst($field['type']),
                'value' => $field['id'],
            ];
        }
        $response['success'] = true;
        $response['fields']   = $fields_data;
        return rest_ensure_response($response);
    }

    /**
     * Processes fields from a WPForms form with columns.
     *
     * Extracts field data from a WPForms form with columns and returns a formatted array
     * of field names and labels, excluding submit buttons and file upload fields.
     *
     * @param array $columns An array of columns containing fields.
     *
     * @return array The formatted array of field data.
     * @since 1.17.4
     */
    private static function process_wp_form_column_fields($columns) {
        $fields_data = [];
    
        foreach ($columns as $column) {
            if (empty($column['fields'])) {
                continue;
            }
    
            foreach ($column['fields'] as $field) {
                if (!empty($field['isHidden']) || $field['type'] === 'submit') {
                    continue;
                }
                $fields_data[] = [
                    'type'  => $field['type'],
                    'label' => !empty($field['label']) ? $field['label'] : ucfirst($field['type']),
                    'value' => $field['id'],
                ];
            }
        }
    
        return $fields_data;
    }
}