<?php
/**
 * Mail Mint Pro
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2025-01-07 15:33:17
 * @modify date 2025-01-07 15:33:17
 * @package MintMailPro\App\Admin\API\Actions\Connector;
 */
namespace MintMailPro\App\Admin\API\Actions\Connector;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Connector handler for Gravity .
 *
 * This class handles API requests related to Gravity forms, including fetching available forms
 * and their fields. It extends the `BaseConnectorHandler` to implement specific functionality for Gravity .
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector
 * @since 1.17.4
 */
class GravityFormsConnectorHandler extends BaseConnectorHandler {

    /**
     * Retrieves a list of Gravity forms.
     *
     * Fetches all Gravity forms based on the search parameter. Returns a formatted array
     * containing the form ID and title for each matching form.
     *
     * @param array $params An array of parameters for the request.
     *
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
    public function get($params) {
        $search = isset($params['search']) ? $params['search'] : '';
        $formatted_forms=[];
        if ( HelperFunctions::is_gform_active() ) {
			if ( class_exists( 'GFFormsModel' ) ) {
				$forms = \GFFormsModel::search_forms( $search, null, 'title', 'ASC', false );

				if ( is_array( $forms ) ) {
					foreach ($forms as $form) {
						$formatted_forms[] = array(
							'value' => $form->id,
							'label' => $form->title,
						);
					}
				}
			}
		}
        $response['success'] = true;
        $response['forms']   = $formatted_forms;
        return $response;
    }

    /**
     * Retrieves fields from a specific Gravity  form.
     *
     * Scans the specified Gravity  form for available fields and returns a formatted array
     * of field names and labels, excluding submit buttons and file upload fields.
     *
     * @param array $params An array of parameters for the request, including the form ID.
     *
     * @return array The response containing the list of fields.
     * @since 1.17.4
     */
    public function fields($params){
        $form_id = isset($params['id']) ? $params['id'] : '';
        if (empty($form_id)) {
			return array();
		}
		$form = \GFAPI::get_form($form_id);

		if ( empty( $form ) || ! is_array( $form ) || ! isset( $form['fields'] ) ) {
			return array();
		}

		$fields = array();
		foreach ($form['fields'] as $field) {
			$fields[] = array(
				'type'  => $field->type,
				'label' => $field->label,
				'value' => $field->id
			);
		}
        $response['success'] = true;
        $response['fields']  = $fields;
        return $response;
    }
}