<?php
/**
 * Mail Mint Pro
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2025-01-08 15:33:17
 * @modify date 2025-01-08 15:33:17
 * @package MintMailPro\App\Admin\API\Actions\Connector;
 */
namespace MintMailPro\App\Admin\API\Actions\Connector;
use MintMail\App\Internal\Automation\HelperFunctions;
use FluentForm\App\Helpers\Helper;

/**
 * Connector handler for fluent forms.
 *
 * This class handles API requests related to Contact Form 7 forms, including fetching available forms
 * and their fields. It extends the `BaseConnectorHandler` to implement specific functionality for Contact Form 7.
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector
 * @since 1.17.4
 */
class FluentFormsConnectorHandler extends BaseConnectorHandler {

    /**
     * Retrieves a list of fluent forms.
     *
     * Fetches all fluent forms based on the search parameter. Returns a formatted array
     * containing the form ID and title for each matching form.
     *
     * @param array $params An array of parameters for the request.
     *
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
	public function get($params) {
		global $wpdb;
		$term  = isset($params['search']) ? $params['search'] : '';
		$table = $wpdb->prefix . 'fluentform_forms';

		// Prepare the search string with wildcards for a LIKE query.
		$search = '%' . $wpdb->esc_like($term) . '%';

		// Query to fetch id as value and name as label.
		$query = $wpdb->prepare("SELECT id AS value, title AS label
			FROM {$table}
			WHERE title LIKE %s
		", $search);

		// Execute the query and return the results.
		$forms = $wpdb->get_results($query, ARRAY_A);
		$response['success'] = true;
		$response['forms']   = $forms;
		return $response;
	}

    /**
     * Retrieves single entry of fluent form.
     *
     *
     * @param array $params An array of parameters for the request.
     *
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
	public function fields( $params ) {
        $id = isset($params['id']) ? $params['id'] : '';
		$form = wpFluent()->table('fluentform_forms')->find($id );

		if (! $form) {
			return array();
		}

		$form_fields = json_decode($form->form_fields);
		$fields_data = array();
		if (empty($form_fields)) {
			return $fields_data;
		}

		$column_field_data = array();
		foreach ($form_fields as $field_key => $fields) {
			if (isset($fields->attributes) && isset($fields->attributes->type) && (('submit' === $fields->attributes->type))) {
				continue;
			}

			if (is_array($fields) && 0 < count($fields)) {
				foreach ($fields as $f_key => $field) {
					$field_label = ! empty($field->settings->label) ? $field->settings->label : '';
					$field_label = empty($field_label) && ! empty($field->settings->admin_field_label) ? $field->settings->admin_field_label : $field_label;
					$field_label = empty($field_label) && ! empty($field->attributes->name) ? $field->attributes->name : $field_label;

					if (isset(
						$field->attributes
					) && isset($field->attributes->type)) {
						$fields_data[$field->attributes->name] = $field_label;
					}

					/** in case fields are in column */
					if (isset(
						$field->columns
					) && ! empty($field->columns)) {
						$column_field_data[] = self::get_fluent_col_fields($field->columns);
					}

					/** if both the setting label and admin field label blank then continue  */
					if ((isset($field->settings->label) && empty($field->settings->label)) && (isset($field->settings->admin_field_label) && empty($field->settings->admin_field_label))) {
						continue;
					}

					/** empty field data then get from setting */
					if (empty($fields_data[$field->attributes->name])) {
						$fields_data[$field->attributes->name] = $field_label;
					}

					/** Getting child fields */
					if (isset(
						$field->fields
					) && ! empty($field->fields)) {
						foreach ($field->fields as $child_field) {
							if (false === $child_field->settings->visible) {
								continue;
							}
							$label = ! empty($child_field->settings->label) ? $child_field->settings->label : '';
							$label = empty($label) && ! empty($child_field->attributes->placeholder) ? $child_field->attributes->placeholder : $label;
							$label = empty($label) && ! empty($child_field->attributes->name) ? $child_field->attributes->name : $label;

							$fields_data[$field->attributes->name . ':' . $child_field->attributes->name] = $label;
							unset($fields_data[$field->attributes->name]);
						}
					}
				}
			}
		}

		if (! empty($column_field_data)) {
			foreach ($column_field_data as $cf_key => $column_field) {
				$fields_data = array_merge($fields_data, $column_field);
			}
		}

		$fields_data        = array_filter($fields_data);
		$transformed_fields = array();
		foreach ($fields_data as $key => $label) {
			$type = strpos($key, ':') !== false ? substr($key, strpos($key, ':') + 1) : $key;
			$transformed_fields[] = array(
				'type' => $type,
				'label' => $label,
				'value' => $type
			);
		}

        $response['success'] = true;
        $response['fields']  = $transformed_fields;
        return $response;
	}

	/**
	 * Get column fields for fluent form.
	 * 
	 * @param array $col_fields An array of column fields.
	 * 
	 * @return array The formatted array of field data.
	 * @since 1.17.4
	 */
    private static function get_fluent_col_fields($col_fields){
		$fields_data = array();

		foreach ($col_fields as $col_key => $col_field) {

			if (! isset(
				$col_field->fields
			) || empty($col_field->fields)) {
				continue;
			}

			foreach ($col_field->fields as $f_key => $field) {
				if (isset($field->attributes) && isset($field->attributes->type)) {
					$field_label = ! empty($field->settings->label) ? $field->settings->label : '';
					$field_label = empty($field_label) && ! empty($field->settings->admin_field_label) ? $field->settings->admin_field_label : $field_label;

					$fields_data[$field->attributes->name] = ! empty($field->attributes->name) ? $field->attributes->name : $field_label;
				}

				if (isset(
					$field->fields
				) && ! empty($field->fields)) {
					foreach ($field->fields as $child_field) {
						if (false === $child_field->settings->visible
						) {
							continue;
						}
						$label = ! empty($child_field->settings->label) ? $child_field->settings->label : '';
						$label = empty($label) && ! empty($child_field->attributes->placeholder) ? $child_field->attributes->placeholder : $label;
						$label = empty($label) && ! empty($child_field->attributes->name) ? $child_field->attributes->name : $label;
						$fields_data[$field->attributes->name . ':' . $child_field->attributes->name] = $label;

						unset($fields_data[$field->attributes->name]);
					}
				}
			}
		}

		return $fields_data;
	}
}