<?php
/**
 * Mail Mint Pro
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2025-01-07 15:33:17
 * @modify date 2025-01-07 15:33:17
 * @package MintMailPro\App\Admin\API\Actions\Connector;
 */
namespace MintMailPro\App\Admin\API\Actions\Connector;
use MintMail\App\Internal\Automation\HelperFunctions;

/**
 * Connector handler for Contact Form 7.
 *
 * This class handles API requests related to Contact Form 7 forms, including fetching available forms
 * and their fields. It extends the `BaseConnectorHandler` to implement specific functionality for Contact Form 7.
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector
 * @since 1.17.4
 */
class ContactFormsConnectorHandler extends BaseConnectorHandler {

    /**
     * Retrieves a list of Contact Form 7 forms.
     *
     * Fetches all Contact Form 7 forms based on the search parameter. Returns a formatted array
     * containing the form ID and title for each matching form.
     *
     * @param array $params An array of parameters for the request.
     *
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
    public function get($params) {
        $search = isset($params['search']) ? $params['search'] : '';

        if (HelperFunctions::is_contact_form_7_active()) {
            $formatted_forms = array();

            $posts = get_posts(
                array(
                    'post_type'   => 'wpcf7_contact_form',
                    'numberposts' => -1,
                    'orderby'     => 'created_at',
                    'order'       => 'DESC',
                    'post_status' => 'publish',
                    's'           => $search,
                )
            );

            if ( is_array($posts) && !empty($posts) ) {
                foreach ($posts as $post) {
                    $formatted_forms[] = array(
                        'value'  => $post->ID,
                        'label'  => $post->post_title,
                    );
                }
            }
            $response['success'] = true;
            $response['forms']   = $formatted_forms;
            return $response;
        }
        return array();
    }

    /**
     * Retrieves fields from a specific Contact Form 7 form.
     *
     * Scans the specified Contact Form 7 form for available fields and returns a formatted array
     * of field names and labels, excluding submit buttons and file upload fields.
     *
     * @param array $params An array of parameters for the request, including the form ID.
     *
     * @return array The response containing the list of fields.
     * @since 1.17.4
     */
    public function fields($params){
        $id = isset($params['id']) ? $params['id'] : '';

        if (HelperFunctions::is_contact_form_7_active()) {
            $form        = \WPCF7_ContactForm::get_instance($id);
            $form_fields = $form->scan_form_tags();
            $fields      = array();

            if (!empty($form_fields)) {
                foreach ($form_fields as $field) {
                    if ('submit' === $field->type || false !== strpos($field->type, 'file')) {
                        continue;
                    }
                    $fields[] = array(
                        'value' => $field->name,
                        'label' => $field->name,
                    );
                }
            }

            $response['success'] = true;
            $response['fields']  = $fields;
            return $response;
        }
        return array();
    }
}