<?php
/**
 * Mail Mint Pro
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector;
 */

namespace MintMailPro\App\Admin\API\Actions\Connector;

use MintMail\App\Internal\Automation\HelperFunctions;
use MintMailPro\Mint_Pro_Helper;
use MailMintPro\Mint\Internal\AbandonedCart\Helper\Common;
use ReflectionClass;

/**
 * Connector handler for bricks forms.
 *
 * This class handles API requests related to Bricksforms, including fetching available forms
 * and their fields.
 *
 * @package MintMailPro\App\Admin\API\Actions\Connector
 * @since 1.17.4
 */
class BricksFormsConnectorHandler extends BaseConnectorHandler {

    /**
     * Retrieves a list of Bricksforms.
     *
     * @param array $params An array of parameters for the request.
     * @return array The response containing the list of forms.
     * @since 1.17.4
     */
    public function get($params) {
        $search = $params['search'] ?? '';

        if (Mint_Pro_Helper::is_bricks_active()) {
            $posts = $this->getBricksPosts();
            $all_forms = [];
            // Retrieve all forms from Bricks posts.
            if (!empty($posts) && is_array($posts)) {
                foreach ($posts as $post) {
                    $post_meta = $this->getBricksPostMeta($post->ID);
                    if (!isset($post_meta) || !is_array($post_meta)) {
                        continue;
                    }
                    foreach ($post_meta as $form) {
                        if ($form['name'] === 'form') {
                            $form_label = !empty($form['label']) ? $form['label'] : 'Untitled form ' . $form['id'];

                            // Apply search filter if $search is provided.
                            if (!empty($search) && stripos($form_label, $search) === false) {
                                continue;
                            }

                            $all_forms[] = [
                                'value'   => $form['id'],
                                'label'   => $form_label,
                            ];
                        }
                    }
                }
            }

            $response['success'] = true;
            $response['forms']   = $all_forms;
            return $response;       
         }
        return [];
    }

    /**
     * Retrieves fields from a specific Bricksform by form ID.
     *
     * @param array $params An array of parameters for the request, including the form ID.
     * @return array The response containing the list of fields.
     * @since 1.17.4
     */
    public function fields($params) {
        $form_id = isset($params['id']) ? sanitize_text_field($params['id']) : '';

        if (empty($form_id)) {
            return [];
        }

        if (!Mint_Pro_Helper::is_bricks_active()) {
            return [];
        }

        $posts = $this->getBricksPosts();

        if (empty($posts) || !is_array($posts)) {
            return [];

        }

        foreach ($posts as $post) {
            $post_meta = $this->getBricksPostMeta($post->ID);

            if (!isset($post_meta) || !is_array($post_meta)) {
                continue;
            }

            $form_data = array_filter(
                $post_meta,
                fn($form) => $form['name'] === 'form' && $form['id'] === $form_id
            );

            if (!empty($form_data)) {
                $form_data = array_pop($form_data);

                $fields = [];
                if (!empty($form_data['settings']['fields']) && is_array($form_data['settings']['fields'])) {
                    foreach ($form_data['settings']['fields'] as $field) {
                        $type = $field['type'] ?? 'text';

                        if ($type === 'upload') {
                            $type = 'file';
                        }

                        $fields[] = [
                            'value' => $field['id'],
                            'type'  => $type,
                            'label' => $field['label'] ?? '',
                        ];
                    }
                }

                return [
                    'success' => true,
                    'fields'  => $fields,
                ];
            }
        }

        return [];

    }

    /**
     * Retrieves all Bricks posts.
     *
     * @return array The list of Bricks posts.
     * @since 1.17.4
     */
    public function getBricksPosts() {
        return $this->invokeCommonMethod('get_bricks_posts');
    }

    /**
     * Retrieves post meta for a Bricks post.
     *
     * @param int $form_id The ID of the form.
     * @return array The post meta for the form.
     * @since 1.17.4
     */
    public function getBricksPostMeta($form_id) {
        return $this->invokeCommonMethod('get_bricks_post_meta', [$form_id]);
    }

    /**
     * Invokes a common method from the Common class.
     *
     * @param string $methodName The name of the method to invoke.
     * @param array $args An array of arguments to pass to the method.
     * @return mixed The result of the method invocation.
     * @since 1.17.4
     */
    private function invokeCommonMethod($methodName, $args = []) {
        $reflection = new ReflectionClass(Common::class);
        $method = $reflection->getMethod($methodName);
        $method->setAccessible(true);
        return $method->invokeArgs(null, $args);
    }
}
