<?php
/**
 * Mail Mint
 *
 * @author [WPFunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2024-01-20 11:03:17
 * @modify date 2024-01-20 11:03:17
 * @package /app/Admin/API/Actions
 */

namespace MintMailPro\Mint\Admin\API\Actions;

use Mint\MRM\API\Actions\Action;
use Mint\MRM\DataBase\Models\CampaignModel;
use MRM\Common\MrmCommon;
use Mint\MRM\Admin\API\Controllers\MessageController;
use Mint\MRM\DataBase\Models\EmailModel;
use MintMail\App\Internal\Automation\AutomationLogModel;
use MailMintPro\App\Utilities\Helper\Analytics;

/**
 * AnalyticsAction class for handling analytics-related actions.
 *
 * This class implements the Action interface and is responsible for defining
 * the behavior related to analytics actions.
 *
 * @since 1.9.0
 */
class AnalyticsAction implements Action {

	/**
	 * Format and retrieve campaign analytics.
	 *
	 * This method takes parameters and retrieves campaign analytics based on
	 * the provided campaign ID.
	 *
	 * @param array $params An array of parameters for retrieving analytics.
	 *
	 * @return array The formatted campaign analytics data.
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_analytics( $params ) {
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$campaign    = CampaignModel::get_campaign_to_analytics( $campaign_id );

		$campaign['scheduled_at'] = MrmCommon::format_campaign_date_time( 'scheduled_at', $campaign );
		$campaign['updated_at']   = MrmCommon::format_campaign_date_time( 'updated_at', $campaign );

		return $campaign;
	}

	/**
	 * Formats and retrieves an overview of campaign metrics based on the provided parameters.
	 *
	 * This function calculates various metrics such as total recipients, delivered and bounced emails,
	 * open rate, click rate, click-to-open rate (CTOR), last 24 hours performance, total unsubscribes,
	 * and order reports. The result is an associative array containing the overview metrics.
	 *
	 * @param array $params An associative array containing the following parameters:
	 *                     - 'campaign_id' (int): The ID of the campaign.
	 *                     - 'email_id' (int): The ID of the email associated with the campaign.
	 *
	 * @return array An associative array containing the analytics keys.
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_overview( $params ) {
		// Extract parameters from the provided array.
		$campaign_id      = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type             = CampaignModel::get_campaign_type( $campaign_id );
		$email_id         = isset( $params['email_id'] ) ? $params['email_id'] : '';
		$total_recipients = CampaignModel::get_campaign_meta_value( $campaign_id, 'total_recipients' );

		// Adjust total recipients for recurring campaigns.
		if ( 'recurring' === $type ) {
			$total_recipients = EmailModel::recurring_campaign_total_recipients( $campaign_id );
		}

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			return AutomationLogModel::prepare_analytics_for_automation_sequence( $campaign_id, $email_id );
		}

		// Calculate delivered and bounced emails.
		$total_delivered = MessageController::prepare_delivered_reports( $email_id, $total_recipients );
		$total_bounced   = MessageController::prepare_bounced_reports( $email_id, $total_recipients );
		$bounced         = isset( $total_bounced['total_bounced'] ) ? $total_bounced['total_bounced'] : '';
		$delivered       = isset( $total_delivered['total_delivered'] ) ? $total_delivered['total_delivered'] : '';

		// Calculate click and open rate.
		$open_rate  = MessageController::prepare_open_rate_reports( $email_id, $bounced, $total_recipients );
		$click_rate = MessageController::prepare_click_rate_reports( $email_id, $bounced, $total_recipients );
		$ctor       = MessageController::prepare_click_to_open_rate_reports( $click_rate, $open_rate );

		// Calculate last 24 hours performance.
		$last_day = MessageController::prepare_last_day_reports( $email_id );

		// Calculate total unsubscribe.
		$unsubscribe = MessageController::prepare_unsubscribe_reports( $email_id, $bounced, $delivered );

		// Calculate order reports.
		$orders = MessageController::prepare_order_reports( $email_id, 'campaign' );

		// Return the final overview array.
		$metrics = array_merge( $total_delivered, $total_bounced, $open_rate, $click_rate, $unsubscribe, $orders, $ctor );

		// Prepare campaign summery.
		$summery = Analytics::prepare_campaign_summery( $campaign_id );

		return array(
			'recipients' => $total_recipients,
			'metrics'    => $metrics,
			'last_day'   => $last_day,
			'summery'    => $summery,
		);
	}

	/**
	 * Formats and retrieves campaign email recipients.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_recipients( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_email_recipients( $campaign_id, $offset, $per_page );
			$total       = Analytics::calculate_total_email_sent_from_automation_sequence( $campaign_id );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_email_recipients( $campaign_id, $offset, $per_page );
		$total       = EmailModel::count_broadcast_email_ids_to_campaign( $campaign_id );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of sent emails.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_sent_to_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_email_recipients( $campaign_id, $offset, $per_page, 'sent' );
			$total       = Analytics::calculate_total_email_sent_from_automation_sequence( $campaign_id, 'sent' );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_email_recipients( $campaign_id, $offset, $per_page, 'sent' );
		$total       = EmailModel::count_broadcast_email_ids_to_campaign( $campaign_id, 'sent' );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of bounced emails.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_bounced_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_email_recipients( $campaign_id, $offset, $per_page, 'failed' );
			$total       = Analytics::calculate_total_email_sent_from_automation_sequence( $campaign_id, 'failed' );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_email_recipients( $campaign_id, $offset, $per_page, 'failed' );
		$total       = EmailModel::count_broadcast_email_ids_to_campaign( $campaign_id, 'failed' );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of opened emails.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_opened_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_sequence_activity_email_list( $campaign_id, $offset, $per_page, 'is_open' );
			$total       = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_open' );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_activity_email_list( $campaign_id, $offset, $per_page, 'is_open' );
		$total       = EmailModel::calculate_open_rate_on_campaign( $campaign_id );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of clicked emails.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_clicked_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_sequence_activity_email_list( $campaign_id, $offset, $per_page, 'is_click' );
			$total       = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_click' );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_activity_email_list( $campaign_id, $offset, $per_page, 'is_click' );
		$total       = EmailModel::calculate_click_rate_on_campaign( $campaign_id );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of unsubscribed emails.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', 'campaign_id', and 'email_id'.
	 * @return array An array containing 'emails' (the retrieved emails), 'total_pages' (the total number of pages), and 'total_count' (the total count of emails).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_unsubscribed_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';
		$type        = CampaignModel::get_campaign_type( $campaign_id );
		$email_id    = isset( $params['email_id'] ) ? $params['email_id'] : '';

		// Handle analytics calculation for automation campaigns separately.
		if ( 'automation' === $type ) {
			$emails      = Analytics::retrieve_automation_sequence_activity_email_list( $campaign_id, $offset, $per_page, 'is_unsubscribe' );
			$total       = EmailModel::count_email_metrics_on_automation_sequence( $campaign_id, 'is_unsubscribe' );
			$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
			return array(
				'emails'      => $emails,
				'total_pages' => $total_pages,
				'total_count' => (int) $total,
			);
		}

		$emails      = Analytics::retrieve_campaign_activity_email_list( $campaign_id, $offset, $per_page, 'is_unsubscribe' );
		$total       = EmailModel::count_unsubscribe_on_campaign( $campaign_id );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'emails'      => $emails,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}

	/**
	 * Formats and retrieves lists of email link click performance.
	 *
	 * This method extracts parameters from the provided array, or uses default values if not set.
	 *
	 * @param array $params An associative array of parameters, which can include 'page', 'per-page', and 'campaign_id'.
	 * @return array An array containing 'results' (the retrieved email link click performance), 'total_pages' (the total number of pages), and 'total_count' (the total count of email link click performance).
	 *
	 * @since 1.9.0
	 */
	public function format_and_retrieve_campaign_email_click_performance_lists( $params ) {
		// Extract parameters or use default values.
		$page        = isset( $params['page'] ) ? $params['page'] : 1;
		$per_page    = isset( $params['per-page'] ) ? $params['per-page'] : 10;
		$offset      = ( $page - 1 ) * $per_page;
		$campaign_id = isset( $params['campaign_id'] ) ? $params['campaign_id'] : '';

		$result = Analytics::retrieve_campaign_click_performance( $campaign_id );

		// Count total number of elements in the response.
		$total = count( $result );

		// Apply offset and limit.
		$response    = array_slice( $result, $offset, $per_page );
		$total_pages = ( 0 !== $per_page ) ? ceil( $total / $per_page ) : 0;
		return array(
			'results'     => $response,
			'total_pages' => $total_pages,
			'total_count' => (int) $total,
		);
	}
}
